class LineNode {
    constructor(x, y) {
        this.x = x;
        this.y = y;
        this.next = null;
    }

    draw(color) {
        stroke(color); 
        if (!this.next) {
            point(this.x, this.y);
            return;
        }
        line(this.x, this.y, this.next.x, this.next.y);
        this.next.draw(color);
    }

    collision(x, y, r) {
        if (!this.next) {
            return false;
        }
        const x1 = this.x;
        const x2 = this.next.x;
        const y1 = this.y;
        const y2 = this.next.y;

        if ((x + r) < Math.min(x1, x2) || (x - r) > Math.max(x1, x2)) {
            return this.next.collision(x, y, r);
        }

        if ((y + r) < Math.min(y1, y2) || (y - r) > Math.max(y1, y2)) {
            return this.next.collision(x, y, r);
        }

        const n = Math.pow((y2 - y1) * x - (x2 - x1) * y + x2 * y1 - y2 * x1, 2);
        const d = Math.pow(y2 - y1, 2) + Math.pow(x2 - x1, 2);

        if ((n / d) < Math.pow(r, 2)) {
            const result = new Vector(y2 - y1, x1 - x2);
            result.normalize();

            return result;
        }
        
        return this.next.collision(x, y, r);
    }
}

class Point {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }

    draw(color) {
        stroke(color)
        point(this.x, this.y);
    }

    collision() {
        return false;
    }
}

class Line {
    constructor(x, y) {
        this.head = new LineNode(x, y);
        this.tail = this.head;
        this.solid = true;
    }

    addNode(x, y) {
        this.tail.next = new LineNode(x, y);
        this.tail = this.tail.next;
    }

    draw(color) {
        this.head.draw(color);
    }

    collision(x, y, r) {
        let current = this.head;
        const result = new Set();
        const threshold = Math.pow(r, 2);
        while (current.next !== null) {
            current = current.next;
            if (!current.next) {
                continue;
            }
            const x1 = current.x;
            const x2 = current.next.x;
            const y1 = current.y;
            const y2 = current.next.y;

            if ((x + r) < Math.min(x1, x2) || (x - r) > Math.max(x1, x2)) {
                continue;
            }

            if ((y + r) < Math.min(y1, y2) || (y - r) > Math.max(y1, y2)) {
                continue;
            }

            const n = Math.pow((y2 - y1) * x - (x2 - x1) * y + x2 * y1 - y2 * x1, 2);
            const d = Math.pow(y2 - y1, 2) + Math.pow(x2 - x1, 2);
            const dist = n / d

            if (dist < threshold) {
                const normal = new Vector(y2 - y1, x1 - x2);
                normal.normalize();
                const side = (x - x1)*(y2 - y1) - (y - y1)*(x2 - x1);
                if (side < 0) {
                    normal.scale(-1);
                }
                
                normal.scale(13 - Math.sqrt(dist));
                result.add(normal);
            }
        }
        return result;
    }
}

class Vector {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }

    add(v) {
        this.x += v.x;
        this.y += v.y;
    }

    magnitude() {
        return Math.pow(Math.pow(this.x, 2) + Math.pow(this.y, 2), 0.5);
    }

    normalize() {
        const magnitude = this.magnitude();
        if (magnitude === 0) {
            return;
        }
        this.scale(1 / magnitude);
    }

    normalized() {
        const magnitude = this.magnitude();
        if (magnitude === 0) {
            return this.copy();
        }
        return this.copy().scale(1 / magnitude);
    }

    copy() {
        return new Vector(this.x, this.y);
    }

    scale(s) {
        this.x *= s;
        this.y *= s;
        return this;
    }

    scaled(s) {
        return new Vector(this.x * s, this.y * s);
    }

    opposite() {
        const result = this.copy();
        result.scale(-1);
        return result;
    }

    static add(v1, v2) {
        return new Vector(v1.x + v2.x, v1.y + v2.y);
    }

    static dot(v1, v2) {
        return v1.x * v2.x + v1.y * v2.y;
    }
}

class PhysicsObject {
    constructor(solid, conservation) {
        this.forces = new Set();
        this.impulses = new Set();
        this.velocity = new Vector(0, 0);
        this.position = new Vector(Infinity, Infinity); 
        this.conservation = conservation;
        this.size = 20; 
        this.color = 255 * this.conservation;
        this.lastCollisions = new Set();
        this.physics = false;
        this.drawing = false;
        this.solid = solid;
        this.contact = new Set();
    }

    reset(x, y) {
        this.position.x = x;
        this.position.y = y;
        this.velocity.scale(0);
        this.physics = false;
        this.drawing = true;
    }

    begin() {
        this.physics = true;
        this.drawing = false;
        this.velocity.x = mouseX - this.position.x;
        this.velocity.y = mouseY - this.position.y;
        this.velocity.scale(0.01);
    }

    draw() {
        if (this.drawing) {
            strokeWeight(2);
            stroke('black');
            this.arrow(this.position.x, this.position.y, mouseX, mouseY);
            strokeWeight(5);
        }
        stroke(0, 0, this.color);
        if (this.solid) {
            stroke(this.color, 0, 0);
        }
        strokeWeight(this.size);
        point(this.position.x, this.position.y);
        strokeWeight(5);
    }

    arrow(x1, y1, x2, y2) {
        line(x1, y1, x2, y2);
        push();
        translate(x2, y2);
        const a = atan2(x1-x2, y2-y1);
        rotate(a);
        line(0, 0, -5, -5);
        line(0, 0, 5, -5);
        pop();
    } 


    resolve() {
        const netImpulse = new Vector(0, 0);
        const netForce = new Vector(0, 0);

        this.forces.forEach((force) => {
            netForce.add(force);
            this.forces.delete(force);
        });

        const collisions = new Set();
        let didCollide = false;
        const compensatoryForce = new Vector(0, 0);
        this.impulses.forEach((collider) => {
            const normal = collider.normal;
            const impulse = normal.normalized();
            const component = Math.abs(Vector.dot(this.velocity, impulse));
            const scaled = impulse.copy();
            scaled.scale(component);
            netImpulse.add(scaled);

            if(Vector.add(impulse, netForce).magnitude() < impulse.magnitude()) {
                const compensatoryComponent = -1 * Vector.dot(netForce, normal);
                compensatoryForce.add(normal.scaled(compensatoryComponent));
            }

            collisions.add(collider.element);
            didCollide = true;
            this.impulses.delete(collider);
        });

        if (!didCollide) {
            this.lastCollisions = new Set();
        }

        if ((netImpulse.x || netImpulse.y) && netImpulse.magnitude() > 0) {
            netImpulse.normalize();
            const projection = Math.abs(Vector.dot(netImpulse, this.velocity));
            netImpulse.scale(projection);

            if (this.lastCollisions.size === collisions.size && [...this.lastCollisions].every(value => collisions.has(value))) {
                netImpulse.scale(0);
                this.velocity.add(compensatoryForce);
            } else {
                netImpulse.scale(this.conservation + 1);
            }
            this.lastCollisions = collisions;
        } else {
            this.velocity.add(compensatoryForce);
        }

        this.velocity.add(netForce);
        this.velocity.add(netImpulse);
        this.position.add(this.velocity);
    }

    applyForce(force) {
        this.forces.add(force);
    }

    applyImpulse(collision) {
        this.impulses.add(collision);
    }

    collision(x, y, r, e) {
        if (! e.solid) {
            return false;
        }
        if (Math.pow(this.position.x - x, 2) + Math.pow(this.position.y - y, 2) > 400) {
            this.contact.delete(e);
            return false;
        }
        const result = new Set();
        const direction = new Vector(x - this.position.x, y - this.position.y);
        direction.normalize();

        if (this.contact.has(e)) {
            return false;
        }

        const netVelocity = Vector.add(this.velocity.scaled(0.25), e.velocity.scaled(-0.5));
        const velocityComponent = Vector.dot(netVelocity, direction);
        direction.scale(velocityComponent);
        result.add(direction);

        this.contact.add(e);
        return result;
    }
}

class ElementManager {
    constructor() {
        this.elements = new Set();
    }

    draw(color) {
        this.elements.forEach((element) => {
            element.draw(color);
        });
    }

    add(element) {
        this.elements.add(element);
    }

    collision(x, y, r, e) {
        const result = new Set();
        for (const element of this.elements) {
            if (!element.solid) {
                continue;
            }
            if (element === e) {
                continue;
            }
            const normals = element.collision(x, y, r, e);
            if (normals) {
                for (const normal of normals) {
                    result.add(new Collision(element, normal));
                }
            }
        }
        if (result.size > 0) {
            return result;
        }
        return false;
    }
}

class Collision {
    constructor(element, normal) {
        this.element = element;
        this.normal = normal;
    }
}

class ElementCreator {
    constructor(movementThreshold) {
        this.element = null;
        this.pMouseX = Infinity;
        this.pMouseY = Infinity;
        this.movementThreshold = movementThreshold;
    }

    draw(color) {
        if (!this.element) {
            return;
        }
        this.element.draw(color);
    }

    mousePressed() {
        this.element = new Line(mouseX, mouseY);
        this.elementManager.add(this.element);
    }

    mouseReleased() {
        if (!this.element) {
            return;
        }
        this.element = null;
    }

    mouseDragged() {
        if (Math.pow((this.pMouseX - mouseX), 2) + Math.pow((this.pMouseY - mouseY), 2) < this.movementThreshold) {
            return;
        }
        this.pMouseX = mouseX;
        this.pMouseY = mouseY;

        if (!this.element) {
            return;
        }
        this.element.addNode(mouseX, mouseY);
    }
}

class Cursor {
    constructor() {
        this.point = new Point(0, 0);
        this.active = false;
    }

    draw() {
        this.point.draw(this.active ? 0 : 128);
    }

    mouseMoved() {
        this.point.x = mouseX;
        this.point.y = mouseY;
    }
}
